<?php
require_once '../includes/config.php';
require_once '../includes/DataBaseObjects.php';

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

header('Content-Type: application/json');

// Start session and check authentication
session_start();
if (!isset($_SESSION['admin_logged_in'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized - Admin session not found']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

$data = json_decode(file_get_contents('php://input'), true);

try {
    $db = new DataBaseObjects();
    
    // Log the incoming data for debugging
    error_log("Incoming data: " . json_encode($data));
    
    // Validate required fields
    if (empty($data['client_id']) || empty($data['currency']) || empty($data['items'])) {
        throw new Exception('Missing required fields');
    }
    
    // Generate invoice number
    $invoice_number = 'INV-' . date('Y') . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
    
    // Calculate totals
    $subtotal = 0;
    foreach ($data['items'] as $item) {
        $subtotal += $item['unit_price'] * $item['quantity'];
    }
    
    // Insert invoice
    $invoice_id = $db->insert('invoices', [
        'client_id' => $data['client_id'],
        'invoice_number' => $invoice_number,
        'issue_date' => $data['issue_date'] ?? date('Y-m-d'),
        'due_date' => $data['due_date'] ?? null,
        'status' => 'issued',
        'currency' => $data['currency'],
        'subtotal' => $subtotal,
        'total' => $subtotal,
        'notes' => $data['notes'] ?? null,
        'created_by' => $_SESSION['admin_id'] ?? null
    ]);
    
    // Insert invoice items and create credits
    foreach ($data['items'] as $item) {
        $item_id = (int)$db->insert('invoice_items', [
            'invoice_id' => $invoice_id,
            'application_id' => $item['application_id'],
            'description' => $item['description'],
            'credit_units' => $item['credit_units'],
            'expiry_date' => $item['expiry_date'] ?? null,
            'unit_price' => $item['unit_price'],
            'quantity' => $item['quantity'] ?? 1,
            'line_subtotal' => $item['unit_price'] * ($item['quantity'] ?? 1)
        ]);
        if (!$item_id) {
            throw new Exception('Failed to insert invoice_items row; missing insert ID');
        }
        
        // Create credit ledger entry
        $db->insert('credits_ledger', [
            'client_id' => $data['client_id'],
            'application_id' => $item['application_id'],
            'invoice_id' => $invoice_id,
            'invoice_item_id' => $item_id,
            'credit_units' => $item['credit_units'],
            'currency' => $data['currency'],
            'expiry_date' => $item['expiry_date'] ?? null,
            'status' => 'active'
        ]);
    }
    
    echo json_encode([
        'success' => true,
        'invoice_id' => $invoice_id,
        'invoice_number' => $invoice_number
    ]);
    
} catch (Exception $e) {
    error_log("Invoices API - Error: " . $e->getMessage());
    error_log("Invoices API - Stack trace: " . $e->getTraceAsString());
    
    http_response_code(400);
    echo json_encode([
        'error' => $e->getMessage(),
        'debug_info' => DEVELOPMENT_MODE ? [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ] : null
    ]);
}
?>
