<?php
// Enable error reporting for development
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include database configuration
require_once("../includes/config.php");

// Set headers for JSON response
header('Content-Type: application/json');

// Start session for admin authentication
session_start();

// Function to validate input parameters
function validateInput($license, $machine_code) {
    if (empty($license) || empty($machine_code)) {
        return false;
    }
    // Add additional validation if needed
    return true;
}

try {
    // Get request method
    $method = $_SERVER['REQUEST_METHOD'];

    switch ($method) {
        case 'POST':
            // Get POST data
            $input = json_decode(file_get_contents('php://input'), true);
            $license = $input['license_key'] ?? '';
            $machine_code = $input['machine_code'] ?? '';
            $settings_json = $input['settings'] ?? '';

            // Validate input
            if (!validateInput($license, $machine_code) || empty($settings_json)) {
                throw new Exception('Invalid input parameters');
            }

            // Verify if the license exists
            $check_sql = "SELECT license_code FROM license WHERE license_code = :license";
            $check_stmt = $db1->prepare($check_sql);
            $check_stmt->execute([':license' => $license]);
            
            if ($check_stmt->rowCount() === 0) {
                throw new Exception('Invalid license key');
            }

            // Insert or update settings
            $sql = "INSERT INTO license_settings (license_code, machine_code, settings_json) 
                    VALUES (:license, :machine_code, :settings_json)
                    ON DUPLICATE KEY UPDATE 
                    settings_json = VALUES(settings_json),
                    updated_at = CURRENT_TIMESTAMP";

            $stmt = $db1->prepare($sql);
            $stmt->execute([
                ':license' => $license,
                ':machine_code' => $machine_code,
                ':settings_json' => $settings_json
            ]);

            echo json_encode([
                'success' => true,
                'message' => 'Settings saved successfully'
            ]);
            break;

        case 'GET':
            // Get query parameters
            $license = $_GET['license_key'] ?? '';
            $machine_code = $_GET['machine_code'] ?? '';

            // Validate input
            if (!validateInput($license, $machine_code)) {
                throw new Exception('Invalid input parameters');
            }

            // Get settings
            $sql = "SELECT settings_json, updated_at 
                    FROM license_settings 
                    WHERE license_code = :license 
                    AND machine_code = :machine_code";

            $stmt = $db1->prepare($sql);
            $stmt->execute([
                ':license' => $license,
                ':machine_code' => $machine_code
            ]);

            $result = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($result) {
                echo json_encode([
                    'success' => true,
                    'data' => [
                        'settings' => $result['settings_json'],
                        'last_updated' => $result['updated_at']
                    ]
                ]);
            } else {
                echo json_encode([
                    'success' => false,
                    'message' => 'No settings found for the provided license and machine code'
                ]);
            }
            break;

        default:
            throw new Exception('Method not allowed');
    }

} catch (Exception $e) {
    error_log("License Settings API - Error: " . $e->getMessage());
    error_log("License Settings API - Stack trace: " . $e->getTraceAsString());
    
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'debug_info' => DEVELOPMENT_MODE ? [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ] : null
    ]);
} catch (PDOException $e) {
    error_log("License Settings API - Database error: " . $e->getMessage());
    error_log("License Settings API - Database stack trace: " . $e->getTraceAsString());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Database error occurred',
        'debug_info' => DEVELOPMENT_MODE ? [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ] : null
    ]);
} 