<?php
//require_once(dirname(__FILE__) . '/config.php');

class DashboardCache {
    private $cachePath;
    private $cacheFile;
    private $cacheLifetime = 3600; // 60 minutes in seconds

    public function __construct() {
        $this->cachePath = dirname(__FILE__) . '/../cache/';
        $this->cacheFile = $this->cachePath . 'dashboard_stats.json';
        
        // Create cache directory if it doesn't exist
        if (!file_exists($this->cachePath)) {
            mkdir($this->cachePath, 0755, true);
        }
    }

    public function getCachedData() {
        if ($this->isCacheValid()) {
            return json_decode(file_get_contents($this->cacheFile), true);
        }
        return null;
    }

    public function updateCache($db, $debug = false) {
        try {
            // Updated query to fix issue with missing companies that only have Enjaz records
            $query = "SELECT 
                c.Id,
                c.CompanyName,
                c.Renew_Date,
                c.alert_enabled,
                c.license_enabled,
                c.account_manager_id,
                am.name AS AccountManagerName,
                COALESCE(co.CNTRY_NAME_LA, 'Unknown') as CountryName,
                COALESCE(passport_counts.passport_count, 0) as total_passports,
                COALESCE(visa_counts.visa_count, 0) as enjaz_passports,
                (SELECT stop_at FROM tbl_clients_applications 
                    WHERE app_id = 4 AND client_id = c.Id 
                    ORDER BY stop_at DESC LIMIT 1) AS package,
                (SELECT stop_at FROM tbl_clients_applications 
                    WHERE app_id = 4 AND client_id = c.Id 
                    ORDER BY stop_at DESC LIMIT 1) - (COALESCE(passport_counts.passport_count, 0) + COALESCE(visa_counts.visa_count, 0)) AS credit
            FROM tbl_clients c
            LEFT JOIN (
                SELECT client_id, COUNT(DISTINCT passport_id) as passport_count 
                FROM passports_1447_2025 
                GROUP BY client_id
            ) as passport_counts ON c.Id = passport_counts.client_id
            LEFT JOIN (
                SELECT client_id, COUNT(DISTINCT visa_id) as visa_count 
                FROM visas_1447
                GROUP BY client_id
            ) as visa_counts ON c.Id = visa_counts.client_id
            LEFT JOIN tbl_countries co ON c.Country = co.CNTRY_ID
            LEFT JOIN tbl_account_managers am ON am.id = c.account_manager_id
            WHERE c.license_enabled = 1
            GROUP BY c.Id
            HAVING package > 0
            ORDER BY (COALESCE(passport_counts.passport_count, 0) + COALESCE(visa_counts.visa_count, 0)) DESC";

            // Enable error mode for debugging
            if ($debug) {
                $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            }
            
            try {
                $stmt = $db->prepare($query);
                $stmt->execute();
                $allClients = $stmt->fetchAll(PDO::FETCH_ASSOC);
            } catch (PDOException $e) {
                throw new Exception("SQL Query Error: " . $e->getMessage());
            }

            // Process renewal dates and calculate days until expiry
            foreach ($allClients as &$client) {
                // Convert Renew_Date to timestamp for calculations
                $renewDate = strtotime($client['Renew_Date']);
                $client['renew_timestamp'] = $renewDate;
                $client['days_until_expiry'] = $renewDate ? ceil(($renewDate - time()) / 86400) : null;
            }

            // Filter for low credit clients (less than or equal to 100 remaining)
            $lowCreditClients = array_filter($allClients, function($client) {
                return isset($client['credit']) && 
                       $client['credit'] <= 100 && 
                       $client['package'] > 100;
            });

            // Sort low credit clients by credit ascending (lowest first)
            usort($lowCreditClients, function($a, $b) {
                return $a['credit'] - $b['credit'];
            });

            // Filter for critical clients and sort by remaining credit descending
            $criticalClients = array_filter($allClients, function($client) {
                if (!isset($client['credit']) || !isset($client['package']) || $client['package'] <= 0) {
                    return false;
                }
                
                $creditPercentage = ($client['credit'] / $client['package']) * 100;
                return ($creditPercentage < 20 || $client['credit'] < 100) && 
                       $client['alert_enabled'] == 1 && 
                       $client['license_enabled'] == 1;
            });

            // Sort critical clients by credit descending
            usort($criticalClients, function($a, $b) {
                return $b['credit'] - $a['credit'];
            });

            // Filter and sort renewal clients (expiring within 90 days)
            $renewalClients = array_filter($allClients, function($client) {
                return isset($client['days_until_expiry']) && 
                       $client['days_until_expiry'] <= 90 && 
                       $client['days_until_expiry'] > 0;
            });

            // Sort by renewal date (earliest first)
            usort($renewalClients, function($a, $b) {
                return $a['days_until_expiry'] - $b['days_until_expiry'];
            });

            // Get inactive clients (haven't used service in last 15+ days)
            $inactiveQuery = "SELECT 
                c.Id,
                c.CompanyName,
                c.Renew_Date,
                COALESCE(co.CNTRY_NAME_LA, 'Unknown') as CountryName,
                COALESCE(passport_counts.passport_count, 0) as total_passports,
                COALESCE(visa_counts.visa_count, 0) as enjaz_passports,
                (SELECT stop_at FROM tbl_clients_applications 
                    WHERE app_id = 4 AND client_id = c.Id 
                    ORDER BY stop_at DESC LIMIT 1) AS package,
                (SELECT stop_at FROM tbl_clients_applications 
                    WHERE app_id = 4 AND client_id = c.Id 
                    ORDER BY stop_at DESC LIMIT 1) - (COALESCE(passport_counts.passport_count, 0) + COALESCE(visa_counts.visa_count, 0)) AS credit,
                (SELECT MAX(Time_stamp) FROM tbl_clients_applications 
                    WHERE Client_id = c.Id 
                    ORDER BY Time_stamp DESC LIMIT 1) AS last_access_date
            FROM tbl_clients c
            LEFT JOIN (
                SELECT client_id, COUNT(DISTINCT passport_id) as passport_count 
                FROM passports_1447_2025 
                GROUP BY client_id
            ) as passport_counts ON c.Id = passport_counts.client_id
            LEFT JOIN (
                SELECT client_id, COUNT(DISTINCT visa_id) as visa_count 
                FROM visas_1447
                GROUP BY client_id
            ) as visa_counts ON c.Id = visa_counts.client_id
            LEFT JOIN tbl_countries co ON c.Country = co.CNTRY_ID
            WHERE c.license_enabled = 1
            HAVING last_access_date IS NOT NULL 
                AND last_access_date < DATE_SUB(NOW(), INTERVAL 15 DAY)
            ORDER BY last_access_date ASC";

            try {
                $stmt = $db->prepare($inactiveQuery);
                $stmt->execute();
                $inactiveClients = $stmt->fetchAll(PDO::FETCH_ASSOC);
            } catch (PDOException $e) {
                throw new Exception("Inactive Clients Query Error: " . $e->getMessage());
            }

            // Process inactive clients to add days since last access
            foreach ($inactiveClients as &$client) {
                $lastAccess = strtotime($client['last_access_date']);
                $client['days_since_last_access'] = $lastAccess ? ceil((time() - $lastAccess) / 86400) : null;
                
                // Determine alert level
                if ($client['days_since_last_access'] >= 30) {
                    $client['alert_level'] = 'critical'; // Red
                } elseif ($client['days_since_last_access'] >= 15) {
                    $client['alert_level'] = 'warning'; // Yellow
                } else {
                    $client['alert_level'] = 'normal';
                }
            }

            // Build insights
            // Country breakdown (top countries by total documents)
            $countryTotals = [];
            foreach ($allClients as $client) {
                $country = $client['CountryName'] ?? 'Unknown';
                $totalDocs = (int)($client['total_passports'] ?? 0) + (int)($client['enjaz_passports'] ?? 0);
                if (!isset($countryTotals[$country])) {
                    $countryTotals[$country] = 0;
                }
                $countryTotals[$country] += $totalDocs;
            }
            arsort($countryTotals);
            $topCountries = [];
            foreach ($countryTotals as $country => $count) {
                $topCountries[] = ['country' => $country, 'total_docs' => (int)$count];
            }
            $topCountries = array_slice($topCountries, 0, 10);

            // Top nationalities breakdown (from passports_1447 table)
            $nationalityQuery = "SELECT 
                co.CNTRY_NAME_LA as nationality_name,
                COUNT(p.passport_id) as total_passports
            FROM passports_1447 p
            LEFT JOIN tbl_countries co ON p.nationality = co.CNTRY_ID            
            GROUP BY p.nationality, co.CNTRY_NAME_LA
            HAVING nationality_name IS NOT NULL
            ORDER BY total_passports DESC
            LIMIT 10";
            
            $topNationalities = [];
            try {
                $stmt = $db->prepare($nationalityQuery);
                $stmt->execute();
                $nationalityResults = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                foreach ($nationalityResults as $nationality) {
                    $topNationalities[] = [
                        'nationality' => $nationality['nationality_name'] ?? 'Unknown',
                        'total_passports' => (int)$nationality['total_passports']
                    ];
                }
            } catch (PDOException $e) {
                // Non-fatal: log and continue
                error_log("Top Nationalities Query Error: " . $e->getMessage());
            }

            // Usage distribution buckets based on package utilization
            $usageBuckets = [
                '0-50' => 0,
                '50-80' => 0,
                '80-90' => 0,
                '90-100' => 0,
            ];
            foreach ($allClients as $client) {
                $package = (float)($client['package'] ?? 0);
                $credit = (float)($client['credit'] ?? 0);
                if ($package <= 0) { continue; }
                $used = max(0, $package - $credit);
                $pct = $package > 0 ? ($used / $package) * 100.0 : 0.0;
                if ($pct < 50) $usageBuckets['0-50']++;
                elseif ($pct < 80) $usageBuckets['50-80']++;
                elseif ($pct < 90) $usageBuckets['80-90']++;
                else $usageBuckets['90-100']++;
            }

            // Renewal buckets
            $renewalBuckets = [
                '0-15' => 0,
                '16-45' => 0,
                '46-90' => 0,
            ];
            foreach ($allClients as $client) {
                $d = $client['days_until_expiry'];
                if ($d === null) continue;
                if ($d <= 15 && $d > 0) $renewalBuckets['0-15']++;
                elseif ($d <= 45) $renewalBuckets['16-45']++;
                elseif ($d <= 90) $renewalBuckets['46-90']++;
            }

            // Recent activity: last 10 clients by last access
            $recentQuery = "SELECT c.Id, c.CompanyName, COALESCE(co.CNTRY_NAME_LA, 'Unknown') as CountryName,
                                (SELECT MAX(Time_stamp) FROM tbl_clients_applications WHERE Client_id = c.Id) AS last_access_date
                             FROM tbl_clients c
                             LEFT JOIN tbl_countries co ON c.Country = co.CNTRY_ID
                             WHERE c.license_enabled = 1
                             ORDER BY last_access_date DESC
                             LIMIT 10";
            $recentActivity = [];
            try {
                $stmt = $db->prepare($recentQuery);
                $stmt->execute();
                $recentActivity = $stmt->fetchAll(PDO::FETCH_ASSOC);
            } catch (PDOException $e) {
                // Non-fatal: log and continue
                error_log("Recent Activity Query Error: " . $e->getMessage());
            }
            foreach ($recentActivity as &$ra) {
                $ts = isset($ra['last_access_date']) ? strtotime($ra['last_access_date']) : null;
                $ra['days_since_last_access'] = $ts ? ceil((time() - $ts) / 86400) : null;
            }

            // Calculate summary data
            $summary = [
                'total_clients' => count($allClients),
                'total_passports' => array_sum(array_column($allClients, 'total_passports')),
                'total_enjaz' => array_sum(array_column($allClients, 'enjaz_passports')),
                'combined_total' => array_sum(array_column($allClients, 'total_passports')) + array_sum(array_column($allClients, 'enjaz_passports')),
                'low_credit' => count($lowCreditClients),
                'critical_credit' => count($criticalClients),
                'renewal_clients' => count($renewalClients),
                'inactive_clients' => count($inactiveClients)
            ];

            // Store in cache
            $cacheData = [
                'timestamp' => time(),
                'data' => $allClients,
                'low_credit_clients' => array_values($lowCreditClients),
                'critical_clients' => array_values($criticalClients),
                'renewal_clients' => array_values($renewalClients),
                'inactive_clients' => array_values($inactiveClients),
                'summary' => $summary,
                'insights' => [
                    'top_countries' => $topCountries,
                    'top_nationalities' => $topNationalities,
                    'usage_distribution' => $usageBuckets,
                    'renewal_buckets' => $renewalBuckets,
                    'recent_activity' => $recentActivity
                ]
            ];

            // Save to cache file
            $this->saveToCache($cacheData);

            return $cacheData;
        } catch (Exception $e) {
            error_log("Cache update error: " . $e->getMessage());
            return false;
        }
    }

    private function isCacheValid() {
        if (!file_exists($this->cacheFile)) {
            return false;
        }

        $cacheData = json_decode(file_get_contents($this->cacheFile), true);
        if (!isset($cacheData['timestamp'])) {
            return false;
        }

        return (time() - $cacheData['timestamp']) < $this->cacheLifetime;
    }

    private function saveToCache($data) {
        try {
            $jsonData = json_encode($data);
            if ($jsonData === false) {
                throw new Exception('Failed to encode cache data: ' . json_last_error_msg());
            }

            $result = file_put_contents($this->cacheFile, $jsonData);
            if ($result === false) {
                throw new Exception('Failed to write to cache file');
            }

            return true;
        } catch (Exception $e) {
            error_log("Cache save error: " . $e->getMessage());
            return false;
        }
    }
} 
